/**
* \file: AilAudioSourceImpl.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Android Auto
*
* \author: I. Hayashi / ADITJ/SW / ihayashi@jp.adit-jv.com
*          D. Girnus  / ADITG/ESM / dgirnus@de.adit-jv.com
*
* \copyright (c) 2016-2017 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <time.h>
#include <sys/prctl.h>
#include <adit_logging.h>

#include <AudioFactory.h>
#include <AudioBackend.h>
#include <AudioTypes.h>

#include <inttypes.h>

#include "AilAudioSourceImpl.h"

LOG_IMPORT_CONTEXT(aauto_audio)

using namespace adit::utility::audio;
using adit::utility::eLogLevel;

namespace adit { namespace aauto
{

AilAudioSourceImpl::AilAudioSourceImpl(AudioSource* inSource)
{
    mSessionId = 0;
    audioSource = inSource;
    mCallbacks = nullptr;

    mShutdown = false;
    mStarted = false;
}

AilAudioSourceImpl::~AilAudioSourceImpl()
{
    if (!mShutdown) {
        shutdown();
    }
}

bool AilAudioSourceImpl::init()
{
    /* read configuration which was set by Application */
    if (true != getCurrConfig()) {
        LOG_ERROR((aauto_audio, "%s, init()  getCurrConfig() failed", mSourceName.c_str()));
        return false;
    }

    /* load AIL backend */
    std::string backendLibName("Alsa");
    mBackend = Factory::Instance()->createBackend(backendLibName,*this);

    /* register callbacks to GalReceiver */
    audioSource->registerCallbacks(this);
    /* set GalReceiver::AudioSource configuration */
    audioSource->setConfig(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);

    mShutdown = false;
    return true;
}

void AilAudioSourceImpl::shutdown()
{
    mShutdown = true;

    // in case we where still running stop and notify
    if (mStarted) {
        stop();
        if (mCallbacks != nullptr)  {
            mCallbacks->microphoneRequestCallback(false, false, false);
        }
    }

    LOGD_DEBUG((aauto_audio, "%s, is down", mSourceName.c_str()));
    return;
}

void AilAudioSourceImpl::setConfigItem(string inKey, string inValue)
{
    LOGD_DEBUG((aauto_audio, "setConfigItem: inKey = %s inValue = %s", inKey.c_str(),inValue.c_str())); 
    mConfig.set(inKey, inValue);
}

void AilAudioSourceImpl::registerCallbacks(IAditAudioSourceCallbacks* inCallbacks)
{
    mCallbacks = inCallbacks;
}


/* IAudioSourceCallbacks */
void AilAudioSourceImpl::ackCallback(int sessionId, uint32_t ack)
{
    if (mSessionId == sessionId && mStarted && mConfig.mVerbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, ackCallback()  sessionId=%d, ack=%d", mSourceName.c_str(), sessionId, ack));
    }
    return;
}

int AilAudioSourceImpl::microphoneRequestCallback(bool open, bool ancEnabled,
        bool ecEnabled, int maxUnacked)
{
    int ret = STATUS_SUCCESS;

    LOGD_DEBUG((aauto_audio, "%s, microphoneRequestCallback() " \
            "open=%d, ancEnabled=%d, ecEnabled=%d, maxUnacked=%d",
            mSourceName.c_str(), open, ancEnabled, ecEnabled, maxUnacked));

    // start audio source
    if (true == open) {
        if (mStarted) {
            LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  already got open!", mSourceName.c_str()));
            return -1;
        }

        mStarted = true;
        bool callbackOK = false;

        if (mCallbacks != nullptr) {
            /* notify Application to start audio capture */
            ret = mCallbacks->microphoneRequestCallback(open, ancEnabled, ecEnabled);
            if (STATUS_SUCCESS == ret) {
                callbackOK = true;
            } else {
                LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  from upper layer failed with ret=%d",
                        mSourceName.c_str(), ret));
            }
        }
        if (callbackOK)
        {
            /* read and set configuration again because Application
             * got set new configuration values due to microphoneRequestCallback(open==true) */
            if (true != getCurrConfig()) {
                LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  getCurrConfig() failed", mSourceName.c_str()));

                if (mCallbacks != nullptr) {
                    mCallbacks->notifyErrorCallback(AUDIO_SOURCE_CONFIGURATION_ERROR);
                }
                ret = -1;
            } else {
                /* set GalReceiver::AudioSource configuration */
                audioSource->setConfig(mConfig.mSampleRate, mConfig.mNumBits, mConfig.mNumChannels);

                /* prepare and start audio capture */
                if (true != start(maxUnacked)) {
                    LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  start() failed!", mSourceName.c_str()));

                    if (mCallbacks != nullptr) {
                        mCallbacks->notifyErrorCallback(AUDIO_SOURCE_START_ERROR);
                    }
                    ret = -1;
                }
            }
        }
    } else { // stop audio source
        if (!mStarted) {
            LOG_ERROR((aauto_audio, "%s, microphoneRequestCallback()  stop without start", mSourceName.c_str()));
            // continue anyway
        }
        mStarted = false;

        /* stop audio capture */
        stop();

        if (mCallbacks != nullptr) {
            /* notify Application to stop audio capture */
            ret = mCallbacks->microphoneRequestCallback(open, ancEnabled, ecEnabled);
        }
    }

    return mSessionId; /* This meaning sessionId = 0 */
}

/* private methods */
uint64_t AilAudioSourceImpl::getTimestamp(void)
{
    struct timespec tp;
    clock_gettime(CLOCK_MONOTONIC, &tp);
    return (tp.tv_sec * NSEC_PER_SEC + tp.tv_nsec) / 1000;
}

bool AilAudioSourceImpl::getCurrConfig()
{
    mSourceName = "AudioSourceMic";

    // read out configuration parameters
    if (true != mConfig.ResultConfig()) {
        LOG_ERROR((aauto_audio, "getCurrConfig()  ResultConfig() failed"));
        return false;
    }

    if ((mConfig.mSampleRate == MICROPHONE_SAMPLING_RATE) && 
        (mConfig.mNumBits == MICROPHONE_BITS_PER_SAMPLE) && 
        (mConfig.mNumChannels == MICROPHONE_CHANNELS))
    {
        /* ok */
    }
    else
    {
        LOG_ERROR((aauto_audio, "%s, Error setting parameter sampling-rate, bits-per-sample, channels",
                mSourceName.c_str()));
        return false;
    }

    return true;
}

bool AilAudioSourceImpl::start(int maxUnacked)
{
    /* prepare AIL */
    /* Size of one period for jitter buffer(converted to number of samples). */
    uint32_t periodSamples = mConfig.mPeriodms*(mConfig.mSampleRate/1000);

    mBackend->setFadeTime( FadeMode::OUT,  StreamDirection::OUT, 0);

    if (!mConfig.mDisablePrio) {
        mBackend->setThreadSched(SCHED_FIFO, mConfig.mThreadPrio);
    }
    if (mConfig.mInitToutms > 0) {
        mBackend->setInitialTimeout(mConfig.mInitToutms);
    }
    std::string playback("");

    /* open AIL - streaming not started */
    AudioError err = mBackend->openStream(mConfig.mDevice, playback, AudioFormat::S16_LE, mConfig.mSampleRate, mConfig.mNumChannels, periodSamples);
    if (err != AudioError::OK) {
        LOG_ERROR((aauto_audio, "%s, openStream() failed, error=%d", mSourceName.c_str(), static_cast<uint32_t>(err)));
        return false;
    } else {
        LOGD_DEBUG((aauto_audio, "%s, periods: %d, stream opened", mSourceName.c_str(), periodSamples));
    }

    /* start streaming -> processing() called */
    err = mBackend->startStream();
    if (err != AudioError::OK) {
        LOG_ERROR((aauto_audio, "%s, startStream() failed, error=%d", mSourceName.c_str(), static_cast<uint32_t>(err)));
        return false;
    } else {
        LOGD_DEBUG((aauto_audio, "%s, stream started", mSourceName.c_str()));
    }

    /* allocate IoBuffer */
    mData = new IoBuffer(periodSamples * (mConfig.mNumBits / 8));
    LOGD_DEBUG((aauto_audio, "%s, I/O Buffer allocated, size: %zu", mSourceName.c_str(), mData->size()));

    return true;
}

void AilAudioSourceImpl::stop()
{
    /* stop streaming */
    AudioError err = mBackend->stopStream();
    if (err != AudioError::OK) {
        LOG_ERROR((aauto_audio, "%s, stopStream() failed, error=%d", mSourceName.c_str(), static_cast<uint32_t>(err)));
    } else {
        LOGD_DEBUG((aauto_audio, "%s, stream stopped", mSourceName.c_str()));
    }

    /* close AIL */
    err = mBackend->closeStream();
    if (err != AudioError::OK) {
        LOG_ERROR((aauto_audio, "%s, closeStream() failed, error=%d", mSourceName.c_str(), static_cast<uint32_t>(err)));
    } else {
        LOG_INFO((aauto_audio, "%s, stream closed", mSourceName.c_str()));
    }
}

/* processing() called by AIL */
AudioState AilAudioSourceImpl::processing(unsigned char *in, unsigned char **out, uint32_t &frames)
{
    memcpy(mData->raw(), in, mData->size());
    uint64_t timestamp = getTimestamp();

    if (mConfig.mVerbose)
    {
        LOGD_VERBOSE((aauto_audio, "%s, processing dataPointer: %p len: %zu. Frames: %d. Timestamp=%" PRIu64 "",
                        mSourceName.c_str(), (uint8_t*)mData->raw(), mData->size(), frames, timestamp));
    }

    /* send AudioSource data */
    audioSource->send(timestamp, mData);

    return AudioState::CONTINUE;
}

/* AIL logging */
void AilAudioSourceImpl::error(const std::string& data) const
{
    LOG_ERROR((aauto_audio, "%s, %s", mSourceName.c_str(), data.c_str()));
}
void AilAudioSourceImpl::warning(const std::string& data) const
{
    LOG_WARN((aauto_audio, "%s, %s", mSourceName.c_str(), data.c_str()));
}
void AilAudioSourceImpl::info(const std::string& data) const
{
    LOG_INFO((aauto_audio, "%s, %s", mSourceName.c_str(), data.c_str()));
}
void AilAudioSourceImpl::debug(const std::string& data) const
{
    LOGD_DEBUG((aauto_audio, "%s, %s", mSourceName.c_str(), data.c_str() ));
}
eLogLevel AilAudioSourceImpl::checkLogLevel() const
{
    return eLogLevel::LL_MAX;
}

void AilAudioSourceImpl::statistics(const StreamStatistics &status)
{
    LOGD_DEBUG((aauto_audio, "%s, %s() time:%" PRIu64 ", flag:%d, xrun count:%u",
              mSourceName.c_str(), __func__, status.time, status.flag, status.xruns.capture));
}

void AilAudioSourceImpl::eostreaming(const AudioError error)
{
    if (error != AudioError::OK) {
        LOG_ERROR((aauto_audio, "%s, %s() Streaming has stopped unexpectedly: %d",
                   mSourceName.c_str(), __func__, (int32_t)error));

        /* notify Application about error and expect that Application stops capturing */
        if (mCallbacks != nullptr) {
            mCallbacks->notifyErrorCallback(AUDIO_SOURCE_READ_ERROR);
        }
    }
}

} } /* namespace adit { namespace aauto */
